<?php
/**
 * Send User Message
 *
 * @package     AutomatorWP\Integrations\BuddyPress\Actions\Send_User_Message
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_BuddyPress_Send_User_Message extends AutomatorWP_Integration_Action {

    public $integration = 'buddypress';
    public $action = 'buddypress_send_user_message';

    /**
     * Store the action result
     *
     * @since 1.0.0
     *
     * @var string $result
     */
    public $result = '';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_action( $this->action, array(
            'integration'       => $this->integration,
            'label'             => __( 'Send a private message to the user', 'automatorwp-pro' ),
            'select_option'     => __( 'Send a <strong>private message</strong> to the user', 'automatorwp-pro' ),
            /* translators: %1$s: Private message. */
            'edit_label'        => sprintf( __( 'Send a %1$s to the user', 'automatorwp-pro' ), '{message}' ),
            /* translators: %1$s: Private message. */
            'log_label'         => sprintf( __( 'Send a %1$s to the user', 'automatorwp-pro' ), '{message}' ),
            'options'           => array(
                'message' => array(
                    'default' => __( 'private message', 'automatorwp-pro' ),
                    'fields' => array(
                        'from' => array(
                            'name' => __( 'From:', 'automatorwp-pro' ),
                            'desc' => __( 'The user ID that sends the message. Leave blank to set the user who completed the automation.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'default' => ''
                        ),
                        'to' => array(
                            'name' => __( 'To:', 'automatorwp-pro' ),
                            'desc' => __( 'The user ID that will receive the message. Leave blank to send the message to the user who completed the automation.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'default' => ''
                        ),
                        'subject' => array(
                            'name' => __( 'Subject:', 'automatorwp-pro' ),
                            'desc' => __( 'Message\'s subject.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'default' => ''
                        ),
                        'content' => array(
                            'name' => __( 'Content:', 'automatorwp-pro' ),
                            'desc' => __( 'Message\'s content.', 'automatorwp-pro' ),
                            'type' => 'wysiwyg',
                            'default' => ''
                        ),
                    )
                )
            ),
        ) );

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation ) {

        $this->result = '';

        // Bail if messages component is not active
        if ( ! bp_is_active( 'messages' ) ) {
            $this->result = __( 'Private messaging component is not active.', 'automatorwp-pro' );
            return;
        }

        // Shorthand
        $from = $action_options['from'];
        $to = $action_options['to'];
        $subject = $action_options['subject'];
        $content = $action_options['content'];

        // Setup from
        if( empty( $from ) ) {
            $from = $user_id;
        }

        if( absint( $from ) === 0 ) {
            $this->result = __( 'Invalid "from" user provided on the action\'s setup.', 'automatorwp-pro' );
            return;
        }

        // Setup to
        if( empty( $to ) ) {
            $to = $user_id;
        }

        if( absint( $to ) === 0 ) {
            $this->result = __( 'Invalid "to" user provided on the action\'s setup.', 'automatorwp-pro' );
            return;
        }

        // Send the message
        messages_new_message( array(
            'sender_id'     => $from,
            'thread_id'     => false,
            'recipients'    => array( $to ),
            'subject'       => $subject,
            'content'       => $content,
            'date_sent'     => bp_core_current_time(),
            'append_thread' => true,
            'is_hidden'     => false,
            'mark_visible'  => false,
            'group_thread'  => false,
            'error_type'    => 'bool',
        ) );

        $this->result = __( 'Message sent successfully.', 'automatorwp-pro' );

    }

    /**
     * Register required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Log meta data
        add_filter( 'automatorwp_user_completed_action_log_meta', array( $this, 'log_meta' ), 10, 5 );

        // Log fields
        add_filter( 'automatorwp_log_fields', array( $this, 'log_fields' ), 10, 5 );

        parent::hooks();

    }

    /**
     * Action custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     *
     * @return array
     */
    public function log_meta( $log_meta, $action, $user_id, $action_options, $automation ) {

        // Bail if action type don't match this action
        if( $action->type !== $this->action ) {
            return $log_meta;
        }

        $log_meta['result'] = $this->result;

        return $log_meta;

    }

    /**
     * Action custom log fields
     *
     * @since 1.0.0
     *
     * @param array     $log_fields The log fields
     * @param stdClass  $log        The log object
     * @param stdClass  $object     The trigger/action/automation object attached to the log
     *
     * @return array
     */
    public function log_fields( $log_fields, $log, $object ) {

        // Bail if log is not assigned to an action
        if( $log->type !== 'action' ) {
            return $log_fields;
        }

        // Bail if action type don't match this action
        if( $object->type !== $this->action ) {
            return $log_fields;
        }

        $log_fields['result'] = array(
            'name' => __( 'Result:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        return $log_fields;

    }

}

new AutomatorWP_BuddyPress_Send_User_Message();